// compile: gcc -O2 -std=c99 mast_c.c -o mast_c -lm
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>

typedef long long ll;

typedef struct {
    ll x;
    ll h;
    int orig;
} Mast;

int cmp_by_x(const void *pa, const void *pb) {
    const Mast *a = (const Mast*)pa;
    const Mast *b = (const Mast*)pb;
    if (a->x < b->x) return -1;
    if (a->x > b->x) return 1;
    return 0;
}

typedef struct {
    int pos;
    ll h;
} HPair;

int cmp_hpair_desc(const void *pa, const void *pb) {
    const HPair *A = (const HPair*)pa;
    const HPair *B = (const HPair*)pb;
    if (A->h > B->h) return -1;
    if (A->h < B->h) return 1;
    return 0;
}

/* iterative segment tree for max and min */
typedef struct {
    int n;      // size (power of two)
    ll *t;      // tree array length 2*n
} SegMax;

typedef struct {
    int n;
    ll *t;
} SegMin;

SegMax makeSegMax(int size) {
    SegMax S;
    int p = 1;
    while (p < size) p <<= 1;
    S.n = p;
    S.t = (ll*)malloc(sizeof(ll) * (2 * p));
    ll neginf = LLONG_MIN / 4;
    for (int i = 0; i < 2*p; ++i) S.t[i] = neginf;
    return S;
}
void freeSegMax(SegMax *S) { free(S->t); S->t = NULL; S->n = 0; }
void segMaxUpdate(SegMax *S, int pos, ll val) {
    int i = pos + S->n;
    if (S->t[i] < val) S->t[i] = val;
    else return; // no need to propagate if not larger (we always store max; we never decrease)
    i >>= 1;
    while (i) {
        ll left = S->t[i<<1], right = S->t[i<<1|1];
        S->t[i] = (left > right ? left : right);
        i >>= 1;
    }
}
ll segMaxQuery(SegMax *S, int l, int r) {
    if (l > r) return LLONG_MIN / 4;
    l += S->n; r += S->n;
    ll res = LLONG_MIN / 4;
    while (l <= r) {
        if (l & 1) { if (S->t[l] > res) res = S->t[l]; l++; }
        if (!(r & 1)) { if (S->t[r] > res) res = S->t[r]; r--; }
        l >>= 1; r >>= 1;
    }
    return res;
}

SegMin makeSegMin(int size) {
    SegMin S;
    int p = 1;
    while (p < size) p <<= 1;
    S.n = p;
    S.t = (ll*)malloc(sizeof(ll) * (2 * p));
    ll inf = LLONG_MAX / 4;
    for (int i = 0; i < 2*p; ++i) S.t[i] = inf;
    return S;
}
void freeSegMin(SegMin *S) { free(S->t); S->t = NULL; S->n = 0; }
void segMinUpdate(SegMin *S, int pos, ll val) {
    int i = pos + S->n;
    if (S->t[i] > val) S->t[i] = val;
    else return; // we only improve (set smaller)
    i >>= 1;
    while (i) {
        ll a = S->t[i<<1], b = S->t[i<<1|1];
        S->t[i] = (a < b ? a : b);
        i >>= 1;
    }
}
ll segMinQuery(SegMin *S, int l, int r) {
    if (l > r) return LLONG_MAX / 4;
    l += S->n; r += S->n;
    ll res = LLONG_MAX / 4;
    while (l <= r) {
        if (l & 1) { if (S->t[l] < res) res = S->t[l]; l++; }
        if (!(r & 1)) { if (S->t[r] < res) res = S->t[r]; r--; }
        l >>= 1; r >>= 1;
    }
    return res;
}

int main() {
    ios:
    int n;
    if (scanf("%d", &n) != 1) return 0;
    Mast *a = (Mast*)malloc(sizeof(Mast) * n);
    for (int i = 0; i < n; ++i) { a[i].orig = i; scanf("%lld", &a[i].x); }
    for (int i = 0; i < n; ++i) scanf("%lld", &a[i].h);

    if (n == 0) return 0;

    // sort by x
    qsort(a, n, sizeof(Mast), cmp_by_x);

    // prepare order by height (store positions in sorted-by-x array)
    HPair *byH = (HPair*)malloc(sizeof(HPair) * n);
    for (int i = 0; i < n; ++i) { byH[i].pos = i; byH[i].h = a[i].h; }
    qsort(byH, n, sizeof(HPair), cmp_hpair_desc);

    // segment trees
    SegMax Tmax = makeSegMax(n);
    SegMin Tmin = makeSegMin(n);

    // answers array in original order
    ll *ans = (ll*)malloc(sizeof(ll) * n);
    for (int i = 0; i < n; ++i) ans[i] = LLONG_MIN; // sentinel for not-set -> means -1 later

    // process groups by equal height descending
    int p = 0;
    while (p < n) {
        ll curH = byH[p].h;
        int q = p;
        // find group end
        while (q < n && byH[q].h == curH) ++q;

        // insert all in [p,q)
        for (int t = p; t < q; ++t) {
            int pos = byH[t].pos;
            ll xv = a[pos].x;
            ll hv = a[pos].h;
            segMaxUpdate(&Tmax, pos, xv + hv);
            segMinUpdate(&Tmin, pos, xv - hv);
        }

        // answer for all in group
        for (int t = p; t < q; ++t) {
            int pos = byH[t].pos;
            ll xi = a[pos].x;
            ll hi = a[pos].h;

            int have = 0;
            ll best = LLONG_MAX; // minimal horizontal distance

            // left side: j < pos
            if (pos - 1 >= 0) {
                ll leftMax = segMaxQuery(&Tmax, 0, pos - 1);
                if (leftMax > LLONG_MIN / 8) {
                    have = 1;
                    ll cand = xi + hi - leftMax; // may be negative
                    if (cand < 0) cand = 0;
                    if (cand < best) best = cand;
                }
            }

            // right side: j > pos
            if (pos + 1 <= n - 1) {
                ll rightMin = segMinQuery(&Tmin, pos + 1, n - 1);
                if (rightMin < LLONG_MAX / 8) {
                    have = 1;
                    ll cand = rightMin - (xi - hi); // may be negative
                    if (cand < 0) cand = 0;
                    if (cand < best) best = cand;
                }
            }

            if (!have) {
                ans[a[pos].orig] = LLONG_MIN; // means -1
            } else {
                ans[a[pos].orig] = best;
            }
        }

        p = q;
    }

    // print answers in original order
    for (int i = 0; i < n; ++i) {
        if (ans[i] == LLONG_MIN) printf("-1\n");
        else printf("%lld\n", ans[i]);
    }

    // free
    free(a);
    free(byH);
    freeSegMax(&Tmax);
    freeSegMin(&Tmin);
    free(ans);
    return 0;
}